package com.ediagnosis.cdr.util;


import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.DateTimeException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.Temporal;
import java.util.Optional;

public class TimeUtil {


    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final Logger log = LoggerFactory.getLogger(TimeUtil.class);

    public static Optional<LocalDateTime> toLocalDateTime(String time) {
        return parseTime(time, LocalDateTime.class, DATE_TIME_FORMATTER);
    }

    public static Optional<String> toString(LocalDateTime time) {

        return formatTime(time,DATE_TIME_FORMATTER);
    }

    public static Optional<LocalDate> toLocalDate(String time) {
        return parseTime(time, LocalDate.class, DATE_FORMATTER);

    }

    public static Optional<String> toString(LocalDate time) {
        return formatTime(time,DATE_FORMATTER);
    }

    private static <T extends Temporal> Optional<T> parseTime(String time, Class<T> temporalClass, DateTimeFormatter formatter) {
        try {
            if (temporalClass.equals(LocalDateTime.class)) {
                LocalDateTime dateTime = LocalDateTime.parse(time, formatter);
                log.debug("time parse success:{}", dateTime);
                return Optional.of(dateTime).map(temporalClass::cast);
            } else if (temporalClass.equals(LocalDate.class)) {
                LocalDate date = LocalDate.parse(time, formatter);
                log.debug("time parse success:{}", date);
                return Optional.of(date).map(temporalClass::cast);
            } else {
                log.error("Unsupported temporal class: {}", temporalClass.getName());
                return Optional.empty();
            }
        } catch (DateTimeParseException e) {
            log.error("time parse error", e);
            return Optional.empty();
        }
    }

    private static Optional<String> formatTime(Temporal time, DateTimeFormatter formatter) {
        try {
            if(time instanceof LocalDateTime dateTime){
                String format = dateTime.format(formatter);
                log.debug("time format success:{}", format);
                return Optional.of(format);
            }else if(time instanceof LocalDate date){
                String format = date.format(formatter);
                log.debug("time format success:{}", format);
                return Optional.of(format);
            }else {
                log.error("Unsupported temporal class: {}", time.getClass().getName());
                return Optional.empty();
            }
        } catch (DateTimeException e){
            log.error("time format error", e);
            return Optional.empty();
        }
    }
}
