package com.ediagnosis.cdr.dataGovernance.indicator.dao;

import com.ediagnosis.cdr.context.TimeUtil;
import com.ediagnosis.cdr.dataGovernance.indicator.dao.entity.IndicatorEntity;
import com.ediagnosis.cdr.dataGovernance.indicator.dao.mapper.IndicatorMapper;
import com.ediagnosis.cdr.dataGovernance.indicator.value.Indicator;
import com.mybatisflex.core.query.QueryChain;
import com.mybatisflex.core.query.QueryWrapper;
import com.mybatisflex.core.update.UpdateChain;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.Optional;

@Component
public class IndicatorDao {


    private final IndicatorMapper indicatorMapper;

    public IndicatorDao(IndicatorMapper indicatorMapper) {
        this.indicatorMapper = indicatorMapper;
    }

    public Optional<Indicator> getIndicator(String code) {

        IndicatorEntity indicatorEntity = QueryChain.of(indicatorMapper)
                .eq(IndicatorEntity::getCode, code)
                .one();

        if (indicatorEntity != null) {
            return Optional.of(toIndicator(indicatorEntity));
        }
        return Optional.empty();
    }


    public void upsertIndicator(Indicator indicator) {
        boolean existed = QueryChain.of(indicatorMapper)
                .eq(IndicatorEntity::getCode, indicator.code())
                .exists();
        if (existed) {
             UpdateChain.of(indicatorMapper)
                    .set(IndicatorEntity::getValue, indicator.value())
                    .set(IndicatorEntity::getUpdateTime,
                            TimeUtil.toString(indicator.updateTime()).orElse(""))
                    .where(IndicatorEntity::getCode)
                    .eq(indicator.code())
                    .update();

        }else {
            indicatorMapper.insert(toEntity(indicator));
        }

    }


    public void deleteIndicator(String code) {
        QueryWrapper queryWrapper = QueryWrapper.create();
        queryWrapper
                .where(IndicatorEntity::getCode)
                .eq(code);
        indicatorMapper.deleteByQuery(queryWrapper);

    }


    private Indicator toIndicator(IndicatorEntity entity) {
        String name = entity.getName();
        String code = entity.getCode();
        String value = entity.getValue();
        String description = entity.getDescription();
        LocalDateTime updateTime = TimeUtil.toLocalDateTime(entity.getUpdateTime()).orElse(null);
        return new Indicator(name, code, value, description, updateTime);
    }

    private IndicatorEntity toEntity(Indicator indicator) {
        IndicatorEntity entity = new IndicatorEntity();
        entity.setName(indicator.name());
        entity.setCode(indicator.code());
        entity.setValue(indicator.value());
        entity.setDescription(indicator.description());
        entity.setUpdateTime(TimeUtil.toString(indicator.updateTime()).orElse(""));
        return entity;
    }
}
