package com.ediagnosis.cdr.dataCollect.dao;


import com.ediagnosis.cdr.dataCollect.dao.dto.*;
import com.ediagnosis.cdr.dataCollect.dao.entity.BusinessDataSource;
import com.ediagnosis.cdr.dataCollect.dao.entity.BusinessDataSystem;
import com.ediagnosis.cdr.dataCollect.dao.entity.BusinessDataTable;
import com.ediagnosis.cdr.dataCollect.dao.entity.BusinessDataTableField;
import com.ediagnosis.cdr.dataCollect.dao.mapper.BusinessDataSourceMapper;
import com.ediagnosis.cdr.dataCollect.dao.mapper.BusinessDataSystemMapper;
import com.ediagnosis.cdr.dataCollect.dao.mapper.BusinessDataTableFieldMapper;
import com.ediagnosis.cdr.dataCollect.dao.mapper.BusinessDataTableMapper;
import com.mybatisflex.core.paginate.Page;
import com.mybatisflex.core.query.QueryWrapper;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.Date;

/**
 * @author monojun
 * @description 业务dao
 * @createTime 2025年08月04日
 */
@Component
@RequiredArgsConstructor
public class BusinessDataDao {

    private final BusinessDataSystemMapper dataSystemMapper;

    private final BusinessDataSourceMapper dataSourceMapper;

    private final BusinessDataTableMapper dataTableMapper;

    private final BusinessDataTableFieldMapper dataTableFieldMapper;

    /**
     * 新增业务系统
     *
     * @param dataSystemDTO dataSystemDTO
     * @return 业务系统主键id
     */
    public Long addDataSystem(BusinessDataSystemDTO dataSystemDTO) {
        BusinessDataSystem dataSystem = new BusinessDataSystem();
        BeanUtils.copyProperties(dataSystemDTO, dataSystem);
        dataSystem.setCreateTime(new Date());
        dataSystem.setUpdateTime(new Date());
        dataSystemMapper.insert(dataSystem);
        return dataSystem.getId();
    }

    /**
     * 更新业务系统
     *
     * @param dataSystemDTO dataSystemDTO
     * @return 业务系统主键id
     */
    public Long updateDataSystem(BusinessDataSystemDTO dataSystemDTO) {
        BusinessDataSystem dataSystem = new BusinessDataSystem();
        BeanUtils.copyProperties(dataSystemDTO, dataSystem);
        dataSystem.setUpdateTime(new Date());
        dataSystemMapper.update(dataSystem);
        return dataSystem.getId();
    }

    /**
     * 新增or更新业务系统
     *
     * @param dataSystemDTO dataSystemDTO
     * @return 业务系统主键id
     */
    public Long insertOrUpdateSystem(BusinessDataSystemDTO dataSystemDTO) {
        BusinessDataSystem dataSystem = new BusinessDataSystem();
        BeanUtils.copyProperties(dataSystemDTO, dataSystem);
        dataSystem.setUpdateTime(new Date());
        dataSystemMapper.insertOrUpdate(dataSystem);
        return dataSystem.getId();
    }

    /**
     * 删除业务系统
     *
     * @param dataSystemId 业务系统id
     */
    public void deleteDataSystem(Long dataSystemId) {
        dataSystemMapper.deleteById(dataSystemId);
    }

    /**
     * 分页查询业务系统信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BusinessDataSystem>
     */
    public Page<BusinessDataSystem> querySystemPage(BusinessDataSystemQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create();
        if (StringUtils.hasText(queryDTO.getName())) {
            queryWrapper.like(BusinessDataSystem::getName, queryDTO.getName());
        }
        queryWrapper.orderBy(BusinessDataSystem::getUpdateTime, false);
        queryWrapper.orderBy(BusinessDataSystem::getId, false);
        return dataSystemMapper.paginate(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper);
    }

    /**
     * 新增数据源
     *
     * @param dataSourceDTO dataSourceDTO
     * @return 数据源主键id
     */
    public Long addDataSource(BusinessDataSourceDTO dataSourceDTO) {
        BusinessDataSource dataSource = new BusinessDataSource();
        BeanUtils.copyProperties(dataSourceDTO, dataSource);
        dataSource.setCreateTime(new Date());
        dataSource.setUpdateTime(new Date());
        dataSourceMapper.insert(dataSource);
        return dataSource.getId();
    }

    /**
     * 更新数据源
     *
     * @param dataSourceDTO dataSourceDTO
     * @return 数据源主键id
     */
    public Long updateDataSource(BusinessDataSourceDTO dataSourceDTO) {
        BusinessDataSource dataSource = new BusinessDataSource();
        BeanUtils.copyProperties(dataSourceDTO, dataSource);
        dataSource.setUpdateTime(new Date());
        dataSourceMapper.update(dataSource);
        return dataSource.getId();
    }

    /**
     * 删除数据源
     *
     * @param dataSourceId 数据源id
     */
    public void deleteDataSource(Long dataSourceId) {
        dataSourceMapper.deleteById(dataSourceId);
    }

    /**
     * 分页查询业务数据源信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BusinessDataSource>
     */
    public Page<BusinessDataSourceDTO> querySourcePage(BusinessDataSourceQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .select("d.*", "ds.name as businessName")
                .from(BusinessDataSource.class).as("d")
                .leftJoin(BusinessDataSystem.class).as("ds")
                .on(BusinessDataSource::getBusinessId, BusinessDataSystem::getId)
                .orderBy(BusinessDataSource::getUpdateTime, false)
                .orderBy(BusinessDataSource::getId, false);
        if (!CollectionUtils.isEmpty(queryDTO.getBusinessIds())) {
            queryWrapper.in(BusinessDataSource::getBusinessId, queryDTO.getBusinessIds());
        }
        if (StringUtils.hasText(queryDTO.getName())) {
            queryWrapper.like(BusinessDataSource::getName, queryDTO.getName());
        }
        if (StringUtils.hasText(queryDTO.getIpAddress())) {
            queryWrapper.like(BusinessDataSource::getIpAddress, queryDTO.getIpAddress());
        }
        if (StringUtils.hasText(queryDTO.getType())) {
            queryWrapper.eq(BusinessDataSource::getType, queryDTO.getType());
        }
        if (queryDTO.getStatus() != null) {
            queryWrapper.eq(BusinessDataSource::getStatus, queryDTO.getStatus());
        }
        return dataSourceMapper.paginateAs(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper, BusinessDataSourceDTO.class);
    }

    /**
     * 新增数据表
     *
     * @param dataTableDTO dataTableDTO
     * @return 数据表主键id
     */
    public Long addDataTable(BusinessDataTableDTO dataTableDTO) {
        BusinessDataTable dataTable = new BusinessDataTable();
        BeanUtils.copyProperties(dataTableDTO, dataTable);
        dataTable.setCreateTime(new Date());
        dataTable.setUpdateTime(new Date());
        dataTableMapper.insert(dataTable);
        return dataTable.getId();
    }

    /**
     * 更新业务系统
     *
     * @param dataTableDTO dataTableDTO
     * @return 数据表主键id
     */
    public Long updateDataTable(BusinessDataTableDTO dataTableDTO) {
        BusinessDataTable dataTable = new BusinessDataTable();
        BeanUtils.copyProperties(dataTableDTO, dataTable);
        dataTable.setUpdateTime(new Date());
        dataTableMapper.update(dataTable);
        return dataTable.getId();
    }

    /**
     * 删除业务系统
     *
     * @param dataTableId 业务系统id
     */
    public void deleteDataTable(Long dataTableId) {
        dataTableMapper.deleteById(dataTableId);
    }

    /**
     * 分页查询业务表信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BusinessDataTable>
     */
    public Page<BusinessDataTableDTO> queryTablePage(BusinessDataTableQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .select("t.*, d.id as sourceId, d.name as sourceName, ds.id as businessId, ds.name as businessName")
                .from(BusinessDataTable.class).as("t")
                .leftJoin(BusinessDataSource.class).as("d")
                .on(BusinessDataTable::getSourceId, BusinessDataSource::getId)
                .leftJoin(BusinessDataSystem.class).as("ds")
                .on(BusinessDataSource::getBusinessId, BusinessDataSystem::getId)
                .orderBy(BusinessDataTable::getUpdateTime, false)
                .orderBy(BusinessDataTable::getId, false);
        if (!CollectionUtils.isEmpty(queryDTO.getBusinessIds())) {
            queryWrapper.in(BusinessDataSource::getBusinessId, queryDTO.getBusinessIds());
        }
        if (StringUtils.hasText(queryDTO.getName())) {
            queryWrapper.like(BusinessDataTable::getName, queryDTO.getName());
        }
        if (!CollectionUtils.isEmpty(queryDTO.getSourceIds())) {
            queryWrapper.in(BusinessDataTable::getSourceId, queryDTO.getSourceIds());
        }
        if (StringUtils.hasText(queryDTO.getTableNo())) {
            queryWrapper.like(BusinessDataTable::getTableNo, queryDTO.getTableNo());
        }
        return dataTableMapper.paginateAs(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper, BusinessDataTableDTO.class);
    }

    /**
     * 分页查询业务表字段信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BusinessDataTableField>
     */
    public Page<BusinessDataTableField> queryFieldPage(BusinessDataTableFieldQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create();
        if (StringUtils.hasText(queryDTO.getFieldName())) {
            queryWrapper.like(BusinessDataTableField::getFieldName, queryDTO.getFieldName());
        }
        if (StringUtils.hasText(queryDTO.getFieldType())) {
            queryWrapper.like(BusinessDataTableField::getFieldType, queryDTO.getFieldType());
        }
        if (queryDTO.getTableId() != null) {
            queryWrapper.eq(BusinessDataTableField::getTableId, queryDTO.getTableId());
        }
        queryWrapper.orderBy(BusinessDataTableField::getUpdateTime, false);
        queryWrapper.orderBy(BusinessDataTableField::getId, false);
        return dataTableFieldMapper.paginate(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper);
    }

}
