package com.ediagnosis.cdr.dataIntegration.dao;

import com.ediagnosis.cdr.dataIntegration.dao.dto.*;
import com.ediagnosis.cdr.dataIntegration.dao.entity.BizDataSource;
import com.ediagnosis.cdr.dataIntegration.dao.entity.BizSystem;
import com.ediagnosis.cdr.dataIntegration.dao.entity.BizTable;
import com.ediagnosis.cdr.dataIntegration.dao.entity.BizTableDefinition;
import com.ediagnosis.cdr.dataIntegration.dao.mapper.BizDataSourceMapper;
import com.ediagnosis.cdr.dataIntegration.dao.mapper.BizSystemMapper;
import com.ediagnosis.cdr.dataIntegration.dao.mapper.BizTableDefinitionMapper;
import com.ediagnosis.cdr.dataIntegration.dao.mapper.BizTableMapper;
import com.mybatisflex.core.paginate.Page;
import com.mybatisflex.core.query.QueryWrapper;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

/**
 * @author monojun
 * @description 业务dao
 * @createTime 2025年08月06日
 */
@Component
@RequiredArgsConstructor
public class BizDataDao {

    private final BizSystemMapper systemMapper;

    private final BizDataSourceMapper dataSourceMapper;

    private final BizTableMapper tableMapper;

    private final BizTableDefinitionMapper tableDefinitionMapper;

    /**
     * 新增or更新业务系统
     *
     * @param bizSystemDTO 业务系统dto
     * @return 业务系统主键id
     */
    public Integer addOrUpdateBizSystem(BizSystemDTO bizSystemDTO) {
        BizSystem bizSystem = new BizSystem();
        BeanUtils.copyProperties(bizSystemDTO, bizSystem);
        bizSystem.setUpdateTime(LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        systemMapper.insertOrUpdate(bizSystem);
        return bizSystem.getId();
    }

    /**
     * 删除业务系统
     *
     * @param dataSystemIds 业务系统id列表
     */
    public void deleteBatchBizSystem(List<Integer> dataSystemIds) {
        systemMapper.deleteBatchByIds(dataSystemIds);
    }

    /**
     * 分页查询业务系统信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BizSystem>
     */
    public Page<BizSystem> querySystemPage(BizSystemQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create();
        if (StringUtils.hasText(queryDTO.getName())) {
            queryWrapper.like(BizSystem::getName, queryDTO.getName());
        }
        queryWrapper.orderBy(BizSystem::getUpdateTime, false);
        queryWrapper.orderBy(BizSystem::getId, false);
        return systemMapper.paginate(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper);
    }

    /**
     * 新增or更新业务数据源
     *
     * @param dataSourceDTO dataSourceDTO
     * @return 数据源主键id
     */
    public Integer addOrUpdateDataSource(BizDataSourceDTO dataSourceDTO) {
        BizDataSource dataSource = new BizDataSource();
        BeanUtils.copyProperties(dataSourceDTO, dataSource);
        dataSource.setUpdateTime(LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        dataSourceMapper.insertOrUpdate(dataSource);
        return dataSource.getId();
    }


    /**
     * 删除数据源
     *
     * @param dataSourceIds 数据源id列表
     */
    public void deleteBatchDataSource(List<Integer> dataSourceIds) {
        dataSourceMapper.deleteBatchByIds(dataSourceIds);
    }

    /**
     * 分页查询业务数据源信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BusinessDataSource>
     */
    public Page<BizDataSourceDTO> querySourcePage(BizDataSourceQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .select("d.*", "ds.name as bizSystemName")
                .from(BizDataSource.class).as("d")
                .leftJoin(BizSystem.class).as("ds")
                .on(BizDataSource::getBizSystemId, BizSystem::getId)
                .orderBy(BizDataSource::getUpdateTime, false)
                .orderBy(BizDataSource::getId, false);
        if (!CollectionUtils.isEmpty(queryDTO.getBizSystemIds())) {
            queryWrapper.in(BizDataSource::getBizSystemId, queryDTO.getBizSystemIds());
        }
        if (StringUtils.hasText(queryDTO.getName())) {
            queryWrapper.like(BizDataSource::getName, queryDTO.getName());
        }
        if (StringUtils.hasText(queryDTO.getIp())) {
            queryWrapper.like(BizDataSource::getIp, queryDTO.getIp());
        }
        if (StringUtils.hasText(queryDTO.getType())) {
            queryWrapper.eq(BizDataSource::getType, queryDTO.getType());
        }
        return dataSourceMapper.paginateAs(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper, BizDataSourceDTO.class);
    }

    /**
     * 新增or更新业务表信息
     *
     * @param tableDTO tableDTO
     * @return 数据表主键id
     */
    public Integer addOrUpdateDataTable(BizTableDTO tableDTO) {
        BizTable bizTable = new BizTable();
        BeanUtils.copyProperties(tableDTO, bizTable);
        bizTable.setUpdateTime(LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        tableMapper.insertOrUpdate(bizTable);
        return bizTable.getId();
    }


    /**
     * 删除业务表信息
     *
     * @param dataTableIds 业务表id列表
     */
    public void deleteBatchDataTable(List<Integer> dataTableIds) {
        tableMapper.deleteBatchByIds(dataTableIds);
    }

    /**
     * 分页查询业务表信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BizDataTableDTO>
     */
    public Page<BizTableDTO> queryTablePage(BizTableQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create()
                .select("t.*, d.id as bizDataSourceId, d.name as bizDataSourceName, ds.id as bizSystemId, ds.name as bizSystemName")
                .from(BizTable.class).as("t")
                .leftJoin(BizDataSource.class).as("d")
                .on(BizTable::getBizDataSourceId, BizDataSource::getId)
                .leftJoin(BizSystem.class).as("ds")
                .on(BizTable::getBizSystemId, BizSystem::getId)
                .orderBy(BizTable::getUpdateTime, false)
                .orderBy(BizTable::getId, false);
        if (!CollectionUtils.isEmpty(queryDTO.getBizSystemIds())) {
            queryWrapper.in(BizTable::getBizSystemId, queryDTO.getBizSystemIds());
        }
        if (!CollectionUtils.isEmpty(queryDTO.getBizDataSourceIds())) {
            queryWrapper.in(BizTable::getBizDataSourceId, queryDTO.getBizDataSourceIds());
        }
        if (StringUtils.hasText(queryDTO.getTableName())) {
            queryWrapper.like(BizTable::getTableName, queryDTO.getTableName());
        }
        if (StringUtils.hasText(queryDTO.getTableCode())) {
            queryWrapper.like(BizTable::getTableCode, queryDTO.getTableCode());
        }
        return tableMapper.paginateAs(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper, BizTableDTO.class);
    }

    /**
     * 分页查询业务表字段信息
     *
     * @param queryDTO 查询dto
     * @return PageInfo<BizTableDefinition>
     */
    public Page<BizTableDefinition> queryFieldPage(BizTableDefinitionQueryDTO queryDTO) {
        QueryWrapper queryWrapper = QueryWrapper.create();
        if (StringUtils.hasText(queryDTO.getFieldName())) {
            queryWrapper.like(BizTableDefinition::getFieldName, queryDTO.getFieldName());
        }
        if (StringUtils.hasText(queryDTO.getFieldType())) {
            queryWrapper.eq(BizTableDefinition::getFieldType, queryDTO.getFieldType());
        }
        if (queryDTO.getTableId() != null) {
            queryWrapper.eq(BizTableDefinition::getTableId, queryDTO.getTableId());
        }
        queryWrapper.orderBy(BizTableDefinition::getId, false);
        return tableDefinitionMapper.paginate(queryDTO.getPageNo(), queryDTO.getPageSize(), queryWrapper);
    }
}
